/*
 * Decompiled with CFR 0.152.
 */
package live.thought.jtminer.stratum;

import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import live.thought.jtminer.util.Console;
import live.thought.jtminer.util.Logger;

public class RejectionMonitor {
    private static final int MAX_REJECTIONS_THRESHOLD = 5;
    private static final long TIME_WINDOW_MINUTES = 5L;
    private static final long CHECK_INTERVAL_SECONDS = 30L;
    private static final long RESET_AFTER_SUCCESS_MINUTES = 2L;
    private final ConcurrentLinkedQueue<Long> rejectionTimestamps = new ConcurrentLinkedQueue();
    private final ScheduledExecutorService scheduler;
    private ScheduledFuture<?> monitorTask;
    private final AtomicBoolean isRunning;
    private volatile long lastSuccessfulShare = 0L;
    private final RestartCallback restartCallback;

    public RejectionMonitor(RestartCallback callback) {
        this.restartCallback = callback;
        this.isRunning = new AtomicBoolean(false);
        this.scheduler = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "Rejection-Monitor");
            t.setDaemon(true);
            return t;
        });
    }

    public void start() {
        if (this.isRunning.compareAndSet(false, true)) {
            Logger.log("Rejection monitor started - will trigger restart if > 5 rejections in 5 minutes");
            this.monitorTask = this.scheduler.scheduleAtFixedRate(this::checkRejectionRate, 30L, 30L, TimeUnit.SECONDS);
            Console.debug("Rejection monitor scheduled successfully", 2);
        }
    }

    public void stop() {
        if (this.isRunning.compareAndSet(true, false)) {
            if (this.monitorTask != null) {
                this.monitorTask.cancel(true);
                this.monitorTask = null;
            }
            Logger.log("Rejection monitor stopped");
            Console.debug("Rejection monitor stopped", 2);
        }
    }

    public void cleanup() {
        this.stop();
        this.rejectionTimestamps.clear();
        if (!this.scheduler.isShutdown()) {
            this.scheduler.shutdownNow();
            try {
                if (!this.scheduler.awaitTermination(2L, TimeUnit.SECONDS)) {
                    Console.debug("Rejection monitor scheduler did not terminate within timeout", 2);
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                Console.debug("Interrupted while waiting for rejection monitor shutdown", 2);
            }
        }
    }

    public void recordRejection() {
        long currentTime = System.currentTimeMillis();
        this.rejectionTimestamps.offer(currentTime);
        Logger.log("Share rejection recorded at " + currentTime + " (total in queue: " + this.rejectionTimestamps.size() + ")");
        this.cleanOldTimestamps(currentTime);
        if (this.isRunning.get()) {
            this.checkRejectionRate();
        }
    }

    public void recordSuccessfulShare() {
        this.lastSuccessfulShare = System.currentTimeMillis();
        Logger.log("Successful share recorded at " + this.lastSuccessfulShare);
        this.checkForStabilityReset();
    }

    private void checkForStabilityReset() {
        int currentRejections;
        if (this.lastSuccessfulShare == 0L) {
            return;
        }
        long currentTime = System.currentTimeMillis();
        long timeSinceLastSuccess = currentTime - this.lastSuccessfulShare;
        if (timeSinceLastSuccess < TimeUnit.MINUTES.toMillis(2L) && (currentRejections = this.getCurrentRejectionCount()) < 3 && currentRejections > 0) {
            Logger.log("Connection appears stable (successful shares + low rejections), considering partial reset");
            this.partialReset();
        }
    }

    private void partialReset() {
        long currentTime = System.currentTimeMillis();
        long shorterCutoff = currentTime - TimeUnit.MINUTES.toMillis(2L);
        int removed = 0;
        while (!this.rejectionTimestamps.isEmpty() && this.rejectionTimestamps.peek() < shorterCutoff) {
            this.rejectionTimestamps.poll();
            ++removed;
        }
        if (removed > 0) {
            Logger.log("Partial reset performed: removed " + removed + " old rejections due to stable connection");
            Console.debug("Connection stabilized - partial rejection reset applied", 2);
        }
    }

    public void resetAfterReconnection() {
        this.rejectionTimestamps.clear();
        this.lastSuccessfulShare = 0L;
        Logger.log("Complete rejection monitor reset after successful reconnection");
        Console.output("@|green Rejection monitor reset - starting fresh after reconnection|@");
    }

    private void checkRejectionRate() {
        if (!this.isRunning.get()) {
            return;
        }
        long currentTime = System.currentTimeMillis();
        this.cleanOldTimestamps(currentTime);
        int recentRejections = this.rejectionTimestamps.size();
        Logger.log("Rejection check: " + recentRejections + " rejections in last 5 minutes (threshold: 5)");
        if (recentRejections >= 5) {
            String reason = "Too many rejections: " + recentRejections + " rejections in 5 minutes (threshold: 5)";
            Logger.log("REJECTION THRESHOLD EXCEEDED: " + reason);
            Console.output("@|red WARNING: " + reason + "|@");
            Console.output("@|yellow Triggering automatic restart to recover connection quality...|@");
            this.stop();
            if (this.restartCallback != null) {
                try {
                    this.restartCallback.triggerRestart(reason);
                }
                catch (Exception e) {
                    Logger.log("Error during restart callback: " + e.getMessage());
                    Console.output("@|red Error triggering restart: " + e.getMessage() + "|@");
                }
            }
        } else if (recentRejections > 0) {
            Console.debug("Rejection rate acceptable: " + recentRejections + "/5 in 5 minutes", 2);
        }
    }

    private void cleanOldTimestamps(long currentTime) {
        long cutoffTime = currentTime - TimeUnit.MINUTES.toMillis(5L);
        while (!this.rejectionTimestamps.isEmpty() && this.rejectionTimestamps.peek() < cutoffTime) {
            Long removed = this.rejectionTimestamps.poll();
            Logger.log("Removed old rejection timestamp: " + String.valueOf(removed) + " (older than 5 minutes)");
        }
    }

    public int getCurrentRejectionCount() {
        this.cleanOldTimestamps(System.currentTimeMillis());
        return this.rejectionTimestamps.size();
    }

    public boolean isRunning() {
        return this.isRunning.get();
    }

    public static interface RestartCallback {
        public void triggerRestart(String var1);
    }
}

