/*
 * Decompiled with CFR 0.152.
 */
package live.thought.jtminer;

import java.util.ArrayList;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicLong;
import live.thought.jtminer.algo.CuckooSolve;
import live.thought.jtminer.stratum.RejectionMonitor;
import live.thought.jtminer.stratum.StratumClient;
import live.thought.jtminer.stratum.StratumException;
import live.thought.jtminer.stratum.StratumSolver;
import live.thought.jtminer.stratum.StratumWork;
import live.thought.jtminer.util.Console;
import live.thought.jtminer.util.Logger;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

public class StratumMiner
implements Observer {
    public static final String VERSION = "v0.7";
    private static final String DEFAULT_PASSWORD = "x";
    private static final long STATS_INTERVAL = 10000L;
    private static final Options options = new Options();
    private static final CommandLineParser gnuParser = new GnuParser();
    private final StratumClient client;
    private final int nThreads;
    private final List<Thread> solverThreads;
    private Thread minThread;
    private volatile boolean running;
    private RejectionMonitor rejectionMonitor;
    private final AtomicLong shares = new AtomicLong(0L);
    private final AtomicLong accepted = new AtomicLong(0L);
    private final AtomicLong rejected = new AtomicLong(0L);
    private volatile long startTime;
    public final AtomicLong cycleCount = new AtomicLong(0L);
    public final AtomicLong solutionCount = new AtomicLong(0L);
    public final AtomicLong cycleCountelasped = new AtomicLong(0L);
    public final AtomicLong solutionCountelapsed = new AtomicLong(0L);
    public final AtomicLong timelapsed = new AtomicLong(0L);
    public final AtomicLong errorCount = new AtomicLong(0L);
    private StratumWork currentWork;

    static {
        StratumMiner.initializeOptions();
    }

    public StratumMiner(String[] args) {
        Logger.log("StratumMiner initialization started");
        this.solverThreads = new CopyOnWriteArrayList<Thread>();
        MinerConfig config = StratumMiner.parseArguments(args);
        this.nThreads = config.threads;
        this.client = this.initializeClient(config);
        this.setWorkHandler();
        this.client.addObserver(this);
        this.rejectionMonitor = new RejectionMonitor(reason -> {
            Logger.log("Restart triggered by rejection monitor: " + reason);
            this.handleRejectionRestart(reason);
        });
        Logger.log("StratumMiner initialization completed with " + this.nThreads + " threads");
    }

    @Override
    public void update(Observable o, Object arg) {
        if (arg instanceof StratumWork) {
            this.currentWork = (StratumWork)arg;
            Logger.log("Work updated to job: " + this.currentWork.getJobId());
        }
    }

    private static void initializeOptions() {
        options.addOption("h", "host", true, "Stratum server host");
        options.addOption("P", "port", true, "Stratum server port");
        options.addOption("u", "user", true, "Worker username");
        options.addOption("p", "pass", true, "Worker password");
        options.addOption("t", "threads", true, "Number of mining threads");
        options.addOption("D", "debug", false, "Enable debug output");
        options.addOption("X", "logging", false, "Enable detailed logging to files");
        options.addOption("H", "help", false, "Display help");
    }

    private static MinerConfig parseArguments(String[] args) {
        try {
            CommandLine cmd = gnuParser.parse(options, args);
            if (cmd.hasOption("H")) {
                StratumMiner.printHelp();
                System.exit(0);
            }
            if (cmd.hasOption("D")) {
                Console.setLevel(2);
            }
            if (cmd.hasOption("X")) {
                Logger.setLoggingEnabled(true);
            }
            StratumMiner.validateRequiredOptions(cmd);
            Logger.log("Miner Init with parameters : host :" + cmd.getOptionValue("h") + " Port : " + cmd.getOptionValue("P") + " User : " + cmd.getOptionValue("u") + " threads : " + StratumMiner.parseThreadCount(cmd));
            return MinerConfig.builder().host(cmd.getOptionValue("h")).port(Integer.parseInt(cmd.getOptionValue("P"))).username(cmd.getOptionValue("u")).password(cmd.getOptionValue("p", DEFAULT_PASSWORD)).threads(StratumMiner.parseThreadCount(cmd)).build();
        }
        catch (NumberFormatException | ParseException e) {
            Logger.log("Error parsing arguments: " + e.getMessage());
            Console.output("@|red Error: " + e.getMessage() + "|@");
            StratumMiner.printHelp();
            throw new IllegalArgumentException("Invalid arguments", e);
        }
    }

    private static void validateRequiredOptions(CommandLine cmd) throws ParseException {
        ArrayList<String> missing = new ArrayList<String>();
        if (!cmd.hasOption("h")) {
            missing.add("host (-h)");
        }
        if (!cmd.hasOption("P")) {
            missing.add("port (-P)");
        }
        if (!cmd.hasOption("u")) {
            missing.add("username (-u)");
        }
        if (!missing.isEmpty()) {
            throw new ParseException("Missing: " + String.join((CharSequence)", ", missing));
        }
    }

    private static String parseHost(String host) {
        return host.startsWith("stratum+tcp://") ? host.substring("stratum+tcp://".length()) : host;
    }

    private static int parseThreadCount(CommandLine cmd) throws ParseException {
        if (!cmd.hasOption("t")) {
            return Runtime.getRuntime().availableProcessors();
        }
        int threads = Integer.parseInt(cmd.getOptionValue("t"));
        if (threads < 1) {
            throw new ParseException("Thread count must be >= 1");
        }
        return threads;
    }

    private StratumClient initializeClient(MinerConfig config) {
        Console.output(String.format("@|cyan JTMiner %s - Stratum Mining Client for mine-n-krush.org|@", VERSION));
        Console.output(String.format("Connecting to %s:%d with worker %s", config.host, config.port, config.username));
        Logger.log("Stratum client created for " + config.host + ":" + config.port + " with worker " + config.username);
        return new StratumClient(config.host, config.port, config.username, config.password);
    }

    private void handleRejectionRestart(String reason) {
        Logger.log("Starting rejection-triggered restart process: " + reason);
        Console.output("@|yellow Initiating restart due to excessive rejections...|@");
        try {
            if (this.client != null && this.client.isConnected()) {
                Logger.log("Disconnecting from pool for rejection restart");
                this.client.disconnect();
            }
            Thread.sleep(2000L);
            if (this.client != null && this.client.getReconnectionHandler() != null) {
                Logger.log("Starting reconnection process after rejection restart");
                this.client.getReconnectionHandler().startReconnection();
            }
        }
        catch (Exception e) {
            Logger.log("Error during rejection restart: " + e.getMessage());
            Console.output("@|red Error during rejection restart: " + e.getMessage() + "|@");
        }
    }

    public void resetRejectionMonitor() {
        if (this.rejectionMonitor != null) {
            this.rejectionMonitor.resetAfterReconnection();
            if (!this.rejectionMonitor.isRunning()) {
                this.rejectionMonitor.start();
            }
        }
    }

    public void handleNewWork(StratumWork work) {
        Logger.log("New work received - Job ID: " + work.getJobId());
        if (this.rejectionMonitor != null && !this.rejectionMonitor.isRunning()) {
            Logger.log("Restarting rejection monitor after receiving new work");
            this.rejectionMonitor.resetAfterReconnection();
            this.rejectionMonitor.start();
        }
        if (this.minThread != null && this.minThread.isAlive()) {
            Logger.log("Stopping previous mining thread");
            this.minThread.interrupt();
            try {
                this.minThread.join(1000L);
                Logger.log("Previous mining thread stopped successfully");
            }
            catch (InterruptedException e) {
                Logger.log("Interrupted while stopping previous mining thread");
                Thread.currentThread().interrupt();
            }
        }
        CountDownLatch stopLatch = new CountDownLatch(1);
        Logger.log("Stopping existing solver threads");
        this.stopSolvers(stopLatch);
        try {
            stopLatch.await();
            Logger.log("All solver threads stopped, starting new work");
        }
        catch (InterruptedException e) {
            Logger.log("Interrupted while waiting for solvers to stop");
            Thread.currentThread().interrupt();
            return;
        }
        this.currentWork = work;
        Thread miningThread = new Thread(() -> {
            Logger.log("Mining thread started for job: " + work.getJobId());
            ArrayList<Thread> threads = new ArrayList<Thread>(this.nThreads);
            ArrayList<StratumSolver> solvers = new ArrayList<StratumSolver>(this.nThreads);
            Console.output("Using " + this.nThreads + " threads.");
            Console.output("Job : " + this.currentWork.getJobId());
            int workCycleCount = 0;
            while (!Thread.currentThread().isInterrupted() && this.currentWork.getJobId().equals(work.getJobId())) {
                Logger.log("Starting work cycle " + ++workCycleCount + " for job: " + work.getJobId());
                try {
                    byte[] header = this.currentWork.getHeader();
                    int temp = 0x1000000;
                    CuckooSolve solve = new CuckooSolve(header, temp, this.nThreads);
                    Logger.log("Creating " + this.nThreads + " solver threads for cycle " + workCycleCount);
                    int n = 0;
                    while (n < this.nThreads) {
                        try {
                            StratumSolver solver = new StratumSolver(this.client, this.currentWork, n, solve, this);
                            Thread t = new Thread((Runnable)solver, "Solver-" + n + "-Cycle-" + workCycleCount);
                            threads.add(t);
                            solvers.add(solver);
                            this.solverThreads.add(t);
                            t.start();
                        }
                        catch (Exception e) {
                            Logger.log("Error creating solver thread " + n + ": " + e.getMessage());
                        }
                        ++n;
                    }
                    Logger.log("All " + threads.size() + " solver threads started for cycle " + workCycleCount);
                    int completedThreads = 0;
                    for (Thread t : threads) {
                        try {
                            t.join();
                            ++completedThreads;
                        }
                        catch (InterruptedException e) {
                            Logger.log("Mining thread interrupted while waiting for solvers in cycle " + workCycleCount);
                            solvers.forEach(StratumSolver::stop);
                            Thread.currentThread().interrupt();
                            return;
                        }
                    }
                    Logger.log("Cycle " + workCycleCount + " completed - " + completedThreads + "/" + threads.size() + " threads finished normally");
                    threads.clear();
                    solvers.clear();
                    this.currentWork.updateNTime();
                }
                catch (Exception e) {
                    Logger.log("Error in work cycle " + workCycleCount + ": " + e.getMessage());
                }
            }
            Logger.log("Mining thread finished for job: " + work.getJobId() + " after " + workCycleCount + " cycles");
            this.solverThreads.clear();
        }, "Mining-Thread-" + work.getJobId());
        Logger.log("Starting mining thread for job: " + work.getJobId());
        miningThread.start();
        this.minThread = miningThread;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void stopSolvers(CountDownLatch latch) {
        Logger.log("Stopping " + this.solverThreads.size() + " active solver threads");
        List<Thread> list = this.solverThreads;
        synchronized (list) {
            int interruptedCount = 0;
            for (Thread t : this.solverThreads) {
                if (!t.isAlive()) continue;
                t.interrupt();
                ++interruptedCount;
            }
            Logger.log("Interrupted " + interruptedCount + " solver threads");
        }
        latch.countDown();
    }

    public void setWorkHandler() {
        Logger.log("Setting up work handler");
        this.client.setWorkHandler(new WorkHandler(){

            @Override
            public void handleNewWork(StratumWork work) {
                StratumMiner.this.handleNewWork(work);
            }

            @Override
            public void stop() {
                Logger.log("Work handler stop requested");
                StratumMiner.this.stop();
            }

            @Override
            public void handleSubmitSuccess() {
                StratumMiner.this.handleSubmitSuccess();
            }

            @Override
            public void handleSubmitFailure(String error) {
                StratumMiner.this.handleSubmitFailure(error);
            }
        });
    }

    public void handleSubmitSuccess() {
        this.accepted.incrementAndGet();
        this.shares.incrementAndGet();
        if (this.rejectionMonitor != null) {
            this.rejectionMonitor.recordSuccessfulShare();
        }
        Logger.log("Share accepted - Total accepted: " + this.accepted.get());
        Console.output("@|green Share accepted|@");
    }

    public void handleSubmitFailure(String error) {
        this.rejected.incrementAndGet();
        this.shares.incrementAndGet();
        if (this.rejectionMonitor != null) {
            this.rejectionMonitor.recordRejection();
        }
        Logger.log("Share rejected: " + error + " - Total rejected: " + this.rejected.get());
        Console.output("@|red Share rejected: " + error + "|@");
    }

    public void handleError(String error) {
        Logger.log("Stratum error occurred: " + error);
        Console.output("@|red Stratum error: " + error + "|@");
    }

    private void printStats() {
        long tempcycleCountelasped = this.cycleCount.get() - this.cycleCountelasped.get();
        long tempsolelasped = this.solutionCount.get() - this.solutionCountelapsed.get();
        long runtime = (System.currentTimeMillis() - this.startTime) / 1000L;
        long elapsed = runtime - this.timelapsed.get();
        long cyclespeed = tempcycleCountelasped / elapsed / 1000L;
        double solspeed = (double)tempsolelasped / (double)elapsed;
        if (runtime % 30L == 0L) {
            Console.output("Instance ID: " + Logger.getInstanceId());
        }
        if (runtime % 10L == 0L) {
            Console.output(String.format("Miner Stats: %d cycles, %d solutions, %d Verify errors, %d kcycles/s, %.2f Sol/s", this.cycleCount.get(), this.solutionCount.get(), this.errorCount.get(), cyclespeed, solspeed));
            Console.output(String.format("Pool Stats: %d shares (%d Accepted, %d Rejected).", this.shares.get(), this.accepted.get(), this.rejected.get()));
            if (runtime % 60L == 0L) {
                Logger.log("Stats - Active threads: " + this.solverThreads.size() + ", Cycles: " + this.cycleCount.get() + ", Solutions: " + this.solutionCount.get() + ", Shares: " + this.shares.get() + " (A:" + this.accepted.get() + "/R:" + this.rejected.get() + ")");
            }
            this.cycleCountelasped.set(this.cycleCount.get());
            this.timelapsed.set(runtime);
            this.solutionCountelapsed.set(this.solutionCount.get());
        }
    }

    public void start() {
        if (this.running) {
            Logger.log("Miner already running, ignoring start request");
            return;
        }
        Logger.log("Starting miner");
        try {
            try {
                this.running = true;
                this.startTime = System.currentTimeMillis();
                if (this.rejectionMonitor != null) {
                    this.rejectionMonitor.start();
                }
                Logger.log("Connecting to stratum server");
                this.client.connect();
                Logger.log("Connected to stratum server, entering main loop");
                while (this.running) {
                    Thread.sleep(1000L);
                    this.printStats();
                }
            }
            catch (StratumException e) {
                Logger.log("Stratum exception in main loop: " + e.getMessage());
                this.handleFatalError(e);
                Logger.log("Main loop ended, performing cleanup");
                this.cleanup();
            }
            catch (InterruptedException e) {
                Logger.log("Main loop interrupted");
                this.handleFatalError(e);
                Logger.log("Main loop ended, performing cleanup");
                this.cleanup();
            }
        }
        finally {
            Logger.log("Main loop ended, performing cleanup");
            this.cleanup();
        }
    }

    public void stop() {
        Logger.log("Stop requested");
        if (this.minThread != null && this.minThread.isAlive()) {
            Logger.log("Interrupting main mining thread");
            this.minThread.interrupt();
        }
        CountDownLatch stopLatch = new CountDownLatch(1);
        this.stopSolvers(stopLatch);
        try {
            stopLatch.await();
            Logger.log("All solver threads stopped");
        }
        catch (InterruptedException e) {
            Logger.log("Interrupted while stopping solvers");
            Thread.currentThread().interrupt();
        }
    }

    private void handleFatalError(Exception e) {
        Logger.log("Fatal error occurred: " + e.getMessage());
        Console.output("@|red Fatal error: " + e.getMessage() + "|@");
        this.cleanup();
        this.stop();
    }

    public void cleanup() {
        Logger.log("Cleanup started - stopping " + this.solverThreads.size() + " threads");
        this.running = false;
        if (this.rejectionMonitor != null) {
            this.rejectionMonitor.cleanup();
        }
        int stoppedThreads = 0;
        for (Thread thread : this.solverThreads) {
            if (thread == null || !thread.isAlive()) continue;
            thread.interrupt();
            try {
                thread.join(1000L);
                ++stoppedThreads;
            }
            catch (InterruptedException e) {
                Logger.log("Interrupted while joining thread: " + thread.getName());
                Thread.currentThread().interrupt();
            }
        }
        Logger.log("Stopped " + stoppedThreads + " threads during cleanup");
        this.solverThreads.clear();
        if (this.client != null) {
            Logger.log("Cleaning up stratum client");
            this.client.cleanup();
        }
        Logger.log("Cleanup completed");
        System.gc();
    }

    private static void printHelp() {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp("StratumMiner", "JTMiner - Stratum Mining Client\n\nOptions:", options, "\nExamples:\n  java -jar jtminer.jar -h pool.example.com -P 3333 -u worker1\n  java -jar jtminer.jar -h pool.example.com -P 3333 -u worker1 -X -D\n\nNote: Use -X to enable detailed file logging, -D for debug console output");
    }

    public static void main(String[] args) {
        Logger.log("Application starting");
        StratumMiner miner = null;
        try {
            try {
                miner = new StratumMiner(args);
                miner.start();
            }
            catch (Exception e) {
                Logger.log("Fatal error in main: " + e.getMessage());
                Console.output("@|red Fatal error: " + e.getMessage() + "|@");
                if (miner != null) {
                    miner.cleanup();
                }
                System.exit(1);
                Logger.log("Application shutting down");
            }
        }
        finally {
            Logger.log("Application shutting down");
        }
    }

    private static class MinerConfig {
        final String host;
        final int port;
        final String username;
        final String password;
        final int threads;

        private MinerConfig(Builder builder) {
            this.host = builder.host;
            this.port = builder.port;
            this.username = builder.username;
            this.password = builder.password;
            this.threads = builder.threads;
        }

        static Builder builder() {
            return new Builder();
        }

        static class Builder {
            private String host;
            private int port;
            private String username;
            private String password;
            private int threads;

            Builder() {
            }

            Builder host(String host) {
                if (host == "") {
                    this.host = "tht.mine-n-krush.org";
                    return this;
                }
                this.host = host;
                return this;
            }

            Builder port(int port) {
                this.port = port;
                return this;
            }

            Builder username(String username) {
                this.username = username;
                return this;
            }

            Builder password(String password) {
                this.password = password;
                return this;
            }

            Builder threads(int threads) {
                this.threads = threads;
                return this;
            }

            MinerConfig build() {
                return new MinerConfig(this);
            }
        }
    }

    public static interface WorkHandler {
        public void handleNewWork(StratumWork var1);

        public void stop();

        public void handleSubmitSuccess();

        public void handleSubmitFailure(String var1);
    }
}

